/*
 * Decompiled with CFR 0.152.
 */
package dan200.computercraft.core.computer;

import com.google.common.annotations.VisibleForTesting;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import dan200.computercraft.ComputerCraft;
import dan200.computercraft.core.computer.ComputerExecutor;
import dan200.computercraft.core.computer.TimeoutState;
import dan200.computercraft.shared.util.ThreadUtils;
import java.util.Objects;
import java.util.TreeSet;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.LockSupport;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public final class ComputerThread {
    private static final ThreadFactory monitorFactory = ThreadUtils.factory("Computer-Monitor");
    private static final ThreadFactory workerFactory = ThreadUtils.factory("Computer-Worker");
    private static final long MONITOR_WAKEUP = TimeUnit.MILLISECONDS.toNanos(100L);
    private static final long DEFAULT_LATENCY = TimeUnit.MILLISECONDS.toNanos(50L);
    private static final long DEFAULT_MIN_PERIOD = TimeUnit.MILLISECONDS.toNanos(5L);
    private static final long LATENCY_MAX_TASKS = DEFAULT_LATENCY / DEFAULT_MIN_PERIOD;
    private static final long REPORT_DEBOUNCE = TimeUnit.SECONDS.toNanos(1L);
    private final ReentrantLock threadLock = new ReentrantLock();
    private static final int RUNNING = 0;
    private static final int STOPPING = 1;
    private static final int CLOSED = 2;
    private final AtomicInteger state = new AtomicInteger(0);
    @Nullable
    private Thread monitor;
    @GuardedBy(value="threadLock")
    private final Worker[] workers;
    @GuardedBy(value="threadLock")
    private int workerCount = 0;
    private final Condition shutdown = this.threadLock.newCondition();
    private final long latency;
    private final long minPeriod;
    private final ReentrantLock computerLock = new ReentrantLock();
    private final Condition workerWakeup = this.computerLock.newCondition();
    private final Condition monitorWakeup = this.computerLock.newCondition();
    private final AtomicInteger idleWorkers = new AtomicInteger(0);
    private final TreeSet<ComputerExecutor> computerQueue = new TreeSet((a, b) -> {
        if (a == b) {
            return 0;
        }
        long at = a.virtualRuntime;
        long bt = b.virtualRuntime;
        if (at == bt) {
            return Integer.compare(a.hashCode(), b.hashCode());
        }
        return at < bt ? -1 : 1;
    });
    private long minimumVirtualRuntime = 0L;

    public ComputerThread(int threadCount) {
        this.workers = new Worker[threadCount];
        int factor = 64 - Long.numberOfLeadingZeros(this.workers.length);
        this.latency = DEFAULT_LATENCY * (long)factor;
        this.minPeriod = DEFAULT_MIN_PERIOD * (long)factor;
    }

    @GuardedBy(value="threadLock")
    private void addWorker(int index) {
        ComputerCraft.log.trace("Spawning new worker {}.", (Object)index);
        this.workers[index] = new Worker(index);
        this.workers[index].owner.start();
        ++this.workerCount;
    }

    @GuardedBy(value="computerLock")
    private void ensureRunning() {
        block6: {
            if (this.monitor != null && (this.idleWorkers.get() > 0 || this.workerCount == this.workers.length)) {
                return;
            }
            this.threadLock.lock();
            try {
                ComputerCraft.log.trace("Possibly spawning a worker or monitor.");
                if (this.monitor == null || !this.monitor.isAlive()) {
                    this.monitor = monitorFactory.newThread(new Monitor());
                    this.monitor.start();
                }
                if (this.idleWorkers.get() != 0 && this.workerCount >= this.workers.length) break block6;
                for (int i = 0; i < this.workers.length; ++i) {
                    if (this.workers[i] != null) continue;
                    this.addWorker(i);
                    break;
                }
            }
            finally {
                this.threadLock.unlock();
            }
        }
    }

    private void advanceState(int newState) {
        int current;
        while ((current = this.state.get()) < newState && !this.state.compareAndSet(current, newState)) {
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean stop(long timeout, TimeUnit unit) throws InterruptedException {
        this.advanceState(1);
        this.threadLock.lock();
        try {
            for (Worker worker : this.workers) {
                ComputerExecutor executor;
                if (worker == null || (executor = worker.currentExecutor.get()) == null) continue;
                executor.timeout.hardAbort();
            }
        }
        finally {
            this.threadLock.unlock();
        }
        this.computerLock.lock();
        try {
            this.workerWakeup.signalAll();
        }
        finally {
            this.computerLock.unlock();
        }
        long timeoutNs = unit.toNanos(timeout);
        this.threadLock.lock();
        try {
            while (this.workerCount > 0) {
                if (timeoutNs <= 0L) {
                    int n = 0;
                    return n != 0;
                }
                timeoutNs = this.shutdown.awaitNanos(timeoutNs);
            }
        }
        finally {
            this.threadLock.unlock();
        }
        this.advanceState(2);
        this.computerLock.lock();
        try {
            this.monitorWakeup.signal();
        }
        finally {
            this.computerLock.unlock();
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void queue(ComputerExecutor executor) {
        this.computerLock.lock();
        try {
            if (this.state.get() != 0) {
                throw new IllegalStateException("ComputerThread is no longer running");
            }
            this.ensureRunning();
            if (executor.onComputerQueue) {
                throw new IllegalStateException("Cannot queue already queued executor");
            }
            executor.onComputerQueue = true;
            this.updateRuntimes(null);
            long newRuntime = this.minimumVirtualRuntime;
            newRuntime = executor.virtualRuntime == 0L ? (newRuntime += this.scaledPeriod()) : (newRuntime -= this.latency / 2L);
            executor.virtualRuntime = Math.max(newRuntime, executor.virtualRuntime);
            boolean wasBusy = this.isBusy();
            this.computerQueue.add(executor);
            this.workerWakeup.signal();
            if (!wasBusy && this.isBusy()) {
                this.monitorWakeup.signal();
            }
        }
        finally {
            this.computerLock.unlock();
        }
    }

    private void updateRuntimes(@Nullable ComputerExecutor current) {
        long minRuntime = Long.MAX_VALUE;
        if (!this.computerQueue.isEmpty()) {
            minRuntime = this.computerQueue.first().virtualRuntime;
        }
        long now = System.nanoTime();
        int tasks = 1 + this.computerQueue.size();
        for (Worker runner : this.workers) {
            ComputerExecutor executor;
            if (runner == null || (executor = runner.currentExecutor.get()) == null) continue;
            minRuntime = Math.min(minRuntime, executor.virtualRuntime += (now - executor.vRuntimeStart) / (long)tasks);
            executor.vRuntimeStart = now;
        }
        if (current != null) {
            minRuntime = Math.min(minRuntime, current.virtualRuntime += (now - current.vRuntimeStart) / (long)tasks);
        }
        if (minRuntime > this.minimumVirtualRuntime && minRuntime < Long.MAX_VALUE) {
            this.minimumVirtualRuntime = minRuntime;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void afterWork(Worker runner, ComputerExecutor executor) {
        Thread currentThread = executor.executingThread.getAndSet(null);
        if (currentThread != runner.owner) {
            ComputerCraft.log.error("Expected computer #{} to be running on {}, but already running on {}. This is a SERIOUS bug, please report with your debug.log.", new Object[]{executor.getComputer().getID(), runner.owner.getName(), currentThread == null ? "nothing" : currentThread.getName()});
        }
        this.computerLock.lock();
        try {
            this.updateRuntimes(executor);
            if (!executor.afterWork() || this.state.get() != 0) {
                return;
            }
            this.computerQueue.add(executor);
            this.workerWakeup.signal();
        }
        finally {
            this.computerLock.unlock();
        }
    }

    long scaledPeriod() {
        int count = 1 + this.computerQueue.size();
        return (long)count < LATENCY_MAX_TASKS ? this.latency / (long)count : this.minPeriod;
    }

    @VisibleForTesting
    public boolean hasPendingWork() {
        return !this.computerQueue.isEmpty();
    }

    @GuardedBy(value="computerLock")
    private boolean isBusy() {
        return this.computerQueue.size() > this.idleWorkers.get();
    }

    private void workerFinished(Worker worker) {
        if (!worker.running.getAndSet(false)) {
            return;
        }
        ComputerCraft.log.trace("Worker {} finished.", (Object)worker.index);
        ComputerExecutor executor = worker.currentExecutor.getAndSet(null);
        if (executor != null) {
            executor.afterWork();
        }
        this.threadLock.lock();
        try {
            --this.workerCount;
            if (this.workers[worker.index] != worker) {
                ComputerCraft.log.error("Worker {} closed, but new runner has been spawned.", (Object)worker.index);
            } else if (this.state.get() == 0 || this.state.get() == 1 && this.hasPendingWork()) {
                this.addWorker(worker.index);
                ++this.workerCount;
            } else {
                this.workers[worker.index] = null;
            }
        }
        finally {
            this.threadLock.unlock();
        }
    }

    private final class Worker
    implements Runnable {
        final int index;
        @Nonnull
        final Thread owner;
        final AtomicBoolean running = new AtomicBoolean(true);
        final AtomicReference<ComputerExecutor> currentExecutor = new AtomicReference<Object>(null);
        AtomicLong lastReport = new AtomicLong(Long.MIN_VALUE);

        Worker(int index) {
            this.index = index;
            this.owner = workerFactory.newThread(this);
        }

        @Override
        public void run() {
            try {
                this.runImpl();
            }
            finally {
                ComputerThread.this.workerFinished(this);
            }
        }

        private void runImpl() {
            block9: while (this.running.get()) {
                ComputerExecutor executor;
                ComputerThread.this.computerLock.lock();
                try {
                    ComputerThread.this.idleWorkers.getAndIncrement();
                    while ((executor = ComputerThread.this.computerQueue.pollFirst()) == null) {
                        if (ComputerThread.this.state.get() >= 1) {
                            return;
                        }
                        ComputerThread.this.workerWakeup.awaitUninterruptibly();
                    }
                }
                finally {
                    ComputerThread.this.idleWorkers.getAndDecrement();
                    ComputerThread.this.computerLock.unlock();
                }
                while (!executor.executingThread.compareAndSet(null, this.owner)) {
                    Thread existing = executor.executingThread.get();
                    if (existing == null) continue;
                    ComputerCraft.log.error("Trying to run computer #{} on thread {}, but already running on {}. This is a SERIOUS bug, please report with your debug.log.", new Object[]{executor.getComputer().getID(), this.owner.getName(), existing.getName()});
                    continue block9;
                }
                if (ComputerThread.this.state.get() >= 1) {
                    executor.queueStop(false, true);
                }
                executor.beforeWork();
                this.currentExecutor.set(executor);
                try {
                    executor.work();
                }
                catch (Exception | LinkageError | VirtualMachineError e) {
                    ComputerCraft.log.error("Error running task on computer #" + executor.getComputer().getID(), e);
                    executor.fastFail();
                }
                finally {
                    ComputerExecutor thisExecutor = this.currentExecutor.getAndSet(null);
                    if (thisExecutor == null) continue;
                    ComputerThread.this.afterWork(this, executor);
                }
            }
        }

        private void reportTimeout(ComputerExecutor executor, long time) {
            if (!ComputerCraft.logComputerErrors) {
                return;
            }
            long now = System.nanoTime();
            long then = this.lastReport.get();
            if (then != Long.MIN_VALUE && now - then - REPORT_DEBOUNCE <= 0L) {
                return;
            }
            if (!this.lastReport.compareAndSet(then, now)) {
                return;
            }
            Thread owner = Objects.requireNonNull(this.owner);
            StringBuilder builder = new StringBuilder().append("Terminating computer #").append(executor.getComputer().getID()).append(" due to timeout (running for ").append((double)time * 1.0E-9).append(" seconds). This is NOT a bug, but may mean a computer is misbehaving.\n").append("Thread ").append(owner.getName()).append(" is currently ").append((Object)owner.getState()).append('\n');
            Object blocking = LockSupport.getBlocker(owner);
            if (blocking != null) {
                builder.append("  on ").append(blocking).append('\n');
            }
            for (StackTraceElement element : owner.getStackTrace()) {
                builder.append("  at ").append(element).append('\n');
            }
            executor.printState(builder);
            ComputerCraft.log.warn(builder.toString());
        }
    }

    private final class Monitor
    implements Runnable {
        private Monitor() {
        }

        @Override
        public void run() {
            ComputerCraft.log.trace("Monitor starting.");
            try {
                this.runImpl();
            }
            finally {
                ComputerCraft.log.trace("Monitor shutting down. Current state is {}.", (Object)ComputerThread.this.state.get());
            }
        }

        private void runImpl() {
            while (ComputerThread.this.state.get() < 2) {
                ComputerThread.this.computerLock.lock();
                try {
                    ComputerThread.this.monitorWakeup.awaitNanos(ComputerThread.this.isBusy() ? ComputerThread.this.scaledPeriod() : MONITOR_WAKEUP);
                }
                catch (InterruptedException e) {
                    ComputerCraft.log.error("Monitor thread interrupted. Computers may behave very badly!", (Throwable)e);
                    break;
                }
                finally {
                    ComputerThread.this.computerLock.unlock();
                }
                this.checkRunners();
            }
        }

        private void checkRunners() {
            for (Worker runner : ComputerThread.this.workers) {
                ComputerExecutor executor;
                if (runner == null || (executor = runner.currentExecutor.get()) == null) continue;
                executor.timeout.refresh();
                long afterStart = executor.timeout.nanoCumulative();
                long afterHardAbort = afterStart - TimeoutState.TIMEOUT - TimeoutState.ABORT_TIMEOUT;
                if (afterHardAbort < 0L) continue;
                executor.timeout.hardAbort();
                executor.abort();
                if (afterHardAbort >= TimeoutState.ABORT_TIMEOUT * 2L) {
                    runner.reportTimeout(executor, afterStart);
                    runner.owner.interrupt();
                    ComputerThread.this.workerFinished(runner);
                    continue;
                }
                if (afterHardAbort < TimeoutState.ABORT_TIMEOUT) continue;
                runner.reportTimeout(executor, afterStart);
                runner.owner.interrupt();
            }
        }
    }
}

